/* ============ */
/* gapchisq.c	*/
/* ============ */
#include <defcodes.h>
#include <miscdefs.h>
#include <gapdefs.h>

static long GapCtrs[MAX_GAP_LEN+1];
/* ==================================================================== */
/* TabulateGapSet - Tabulates Complete Set of Gaps			*/
/* ==================================================================== */
/* -------------------------------------------------------------------- */
/* This code reflects the algorithm given by D. E. Knuth, The Art of	*/
/* Computer Programming, Vol. 2, Seminumerical Algorithms, Second Ed.,	*/
/* Addison-Wesley, Reading (1981), pp. 60-61, Algorithm G.		*/ 
/* -------------------------------------------------------------------- */
static void
TabulateGapSet(GAP_DATA_STRU * GapData)
{
    long    GapCt = 0;
    int     GapSize;

    /* ----------------------------------- */
    /* Clear Gap Counters & Variate Counts */
    /* ----------------------------------- */
    memset(GapCtrs, 0, sizeof(GapCtrs));
    GapData->TotNumGen    = 0;
    GapData->ActMaxPerGap = 0;

    do
    {
	/* ---------------- */
	/* Generate New Gap */
	/* ---------------- */
	GapSize = GenGapData(GapData);

	if (GapSize > GapData->ActMaxPerGap)
	{
	    GapData->ActMaxPerGap = GapSize;
	}

	/* ------------------------------ */
	/* Bump Number Variates Generated */
	/* ------------------------------ */
	GapData->TotNumGen += GapSize + 1;

	/* --------------------- */
	/* If No Gap Found, Punt */
	/* --------------------- */
	if (!GapData->CallStatusOK)
	{
	    break;
	}

	if (GapSize > GapData->MaxGapLen)
	{
	    GapSize = GapData->MaxGapLen;
	}

	++GapCtrs[GapSize];
	++GapCt;
    }
    while (GapCt < GapData->NumGaps);
}
/* ==================================================================== */
/* CalcGapChiSq - Gets Chi-Square Statistic for Gap Test		*/
/* ==================================================================== */
void
CalcGapChiSq(GAP_DATA_STRU * GapData)
{
    TabulateGapSet(GapData);

    if (GapData->CallStatusOK)
    {
	int	j;

	/* ----------------------------------- */
	/* Lump Data in Categories as Required */
	/* ----------------------------------- */
	for (j = 0; j < GapData->NotEnough; ++j)
	{
	    GapCtrs[j + 1] += GapCtrs[j];
	}

	GapData->GapChiSq = 0;

	for (j = 0; j < GapData->NumCategories; ++j)
	{
	    if (GapCtrs[j])
	    {
		GapData->GapChiSq +=
		    SQR((double) GapCtrs[j]) / GapData->CellExpect[j];
	    }
	    P(printf("\tGapChiSq = %.15e\n", GapData->GapChiSq));
	}

	GapData->GapChiSq -= (double)GapData->NumGaps;

#	if defined(PRT_GAP_CTRS)
            {
		static x = 3;
		if (x >= 1)
		{
		    for (j = 0; j < GapData->NumCategories; ++j)
		    {
	    		(printf("GapCtrs[%2d] = %ld\n", j, GapCtrs[j]));
		    }
		    --x;
		}
	    }
#	endif
    }
}
